#################################
# Compute consistent S estimators <ith Tukey's biweight function under
# - normality
# - multivariate Student 
# - multivariate power exponential
# /!\ The elliptical distributions are defined such that COV(X)= sigma, 
#######################################################################

# Files  needed to run this script : 
# -fastS_normality
# -fastS_consistency

#---------- Main function 

consistS<-function(data,bdp,dist,df){
  # Function to compute consistent S location and scatter estimates
  
  # INPUTS:
  # data: data matrix of dimension n times p
  # bdp : breakdown point for the MCD, RMCD or S estimators
  # dist : assumed distribution used for the computation of consistency factors 
  #     'norm' : multivariate normal
  #     'stud' : multivariate Student
  #     'powerexp' : multivariate power exponential distribution
  #     /!\ The elliptical distributions are defined such that COV(X)= sigma, 
  # df : degree of freedom for the Student or powerexp distribution 
  #       NB: for the power exp, df=1 corresponds to the normal distribution
  # OUTPUTS:
  # center : S location estimate
  # cov: S scater estimate
  
  data<-as.matrix(data)
  p<-ncol(data)
  n<-nrow(data)
  
  if(n<p) stop("the sample size needs to be larger than the dimension")
  if((dist=="stud" | dist=="powerexp") & is.null(df)) stop("you have to specify a degree of freedom")
  
  if(dist=="norm"){
    Sestim<-fastSmulti_location(data,bdp=bdp)
    
  }else if(dist=="stud"){
    c0<-uniroot(Sint2_stud,interval=c(1,50),p=p,df=df,bdp=bdp)$root
    b0<-bfct_stud(c0,p,df)
    Sestim<-fastSmulti_consist(data,bdp=bdp,c0=c0,b0=b0)
    
  }else if(dist=="powerexp"){
    c0<-uniroot(Sint2_mvexp,interval=c(1,50),p=p,df=df,bdp=bdp)$root
    b0<-bfct_mvexp(c0,p,df)
    Sestim<-fastSmulti_consist(data,bdp=bdp,c0=c0,b0=b0) 
  }else stop("This is not  valid dist value. The possible values are: 'norm', 'stud' or 'powerexp'")
  return(list(center=as.vector(Sestim$Mu), cov=Sestim$Sigma))
}


#----------- Auxiliary functions

# Functions involved in the computation of c0 - Student
Sfct_stud<-function(c,e,p,df){
  K=gamma(0.5*(p+df))/((df-2)^(0.5*p)*gamma(0.5*df)*pi^(0.5*p));
  constant= 2* pi^(0.5*p) /gamma(p*0.5) *K;
  return(constant*integrate(stud_integrand,lower=0,upper=c,e=e,p=p,df=df)$value)
}
bfct_stud<-function(c,p,df){
  return(0.5*Sfct_stud(c,e=2,p=p,df=df) - 0.5/c^2*Sfct_stud(c,e=4,p=p,df=df)+1/(6*c^4)*Sfct_stud(c,e=6,p=p,df=df) + c^2/6*(1-Sfct_stud(c,e=0,p=p,df=df)))
}
Sint2_stud<-function(c,p,df,bdp) bfct_stud(c,p,df)-bdp*c^2/6


# Functions involved in the computation of c0 - mvexp
Sfct_mvexp<-function(c,e,p,df){
  a=(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
  K=df*gamma(0.5*p)/(pi^(0.5*p)*gamma(p/(2*df)))*a^(p/(2*df))
  constant<-2* pi^(0.5*p) /gamma(p*0.5) *K;
  return(constant *integrate(mvexp_integrand,lower=0,upper=c,e=e,p=p,df=df)$value)
}
bfct_mvexp<-function(c,p,df){
  return(0.5*Sfct_mvexp(c,e=2,p=p,df=df) - 0.5/c^2*Sfct_mvexp(c,e=4,p=p,df=df)+1/(6*c^4)*Sfct_mvexp(c,e=6,p=p,df=df) + c^2/6*(1-Sfct_mvexp(c,e=0,p=p,df=df)))
}
Sint2_mvexp<-function(c,p,df,bdp) bfct_mvexp(c,p,df)-bdp*c^2/6

